<?php
/**
 * Admin Page Template for FFL-DB Connector
 * 
 * @package FFL-DB-Connector
 */

if (!defined('ABSPATH')) {
    exit;
}

// Variables available:
// $api_key - Current API key
// $is_verified - Whether API is verified
// $settings - Plugin settings array
?>
<div class="wrap">
    <h1><?php esc_html_e('FFL-DB Connector', 'ffl-db-connector'); ?></h1>
    
    <!-- API Configuration -->
    <div class="card">
        <h2><?php esc_html_e('API Configuration', 'ffl-db-connector'); ?></h2>
        
        <table class="form-table">
            <tr>
                <th scope="row">
                    <label for="ffldb_api_key"><?php esc_html_e('API Key', 'ffl-db-connector'); ?></label>
                </th>
                <td>

		<input type="password" 
       		id="ffldb_api_key" 
       		name="ffldb_api_key" 
       		value="<?php echo esc_attr($api_key); ?>" 
       		class="regular-text" 
       		placeholder="<?php esc_attr_e('Enter your FFL-DB API key', 'ffl-db-connector'); ?>" 
		style="margin-bottom: 10px;" />
		    <button type="button" class="button" id="ffldb_toggle_key" style="margin-right: 5px;"><?php esc_html_e('Show', 'ffl-db-connector'); ?></button>
                    <button type="button" class="button button-primary" id="ffldb_verify_key"><?php esc_html_e('Verify API Key', 'ffl-db-connector'); ?></button>
                    
                    <?php if ($is_verified): ?>
                    <span class="dashicons dashicons-yes-alt" style="color: #46b450; margin-left: 10px;"></span>
                    <span style="color: #46b450;"><?php esc_html_e('Verified', 'ffl-db-connector'); ?></span>
                    <?php endif; ?>
                    
                    <p class="description">
                        <?php printf(
                            __('Get your API key from <a href="%s" target="_blank">FFL-DB.com Account Settings</a>', 'ffl-db-connector'),
                            'https://ffl-db.com/api/'
                        ); ?>
                    </p>
                    <div id="ffldb_verify_status"></div>
                </td>
            </tr>
        </table>
    </div>
    
    <?php if ($is_verified): ?>
    <!-- Cache Settings -->
    <div class="card">
        <h2><?php esc_html_e('Cache Settings', 'ffl-db-connector'); ?></h2>
        
        <p class="description">
            <?php esc_html_e('FFL verification results are cached to minimize API calls.', 'ffl-db-connector'); ?>
        </p>
        
        <p style="margin-top: 15px;">
            <button type="button" class="button" id="ffldb_clear_cache"><?php esc_html_e('Clear All Cached Data', 'ffl-db-connector'); ?></button>
            <span class="description" style="margin-left: 10px;"><?php esc_html_e('Only use this if you need to force-refresh cached FFLs', 'ffl-db-connector'); ?></span>
        </p>
    </div>
    
    <!-- Test FFL Lookup -->
    <div class="card">
        <h2><?php esc_html_e('Test FFL Lookup', 'ffl-db-connector'); ?></h2>
        
        <table class="form-table">
            <tr>
                <th scope="row">
                    <label for="ffldb_test_number"><?php esc_html_e('FFL Number', 'ffl-db-connector'); ?></label>
                </th>
                <td>
                    <input type="text" 
       			id="ffldb_test_number" 
       			placeholder="<?php esc_attr_e('e.g., 1-54-001-01-5K-04379', 'ffl-db-connector'); ?>" 
       			class="regular-text"
       			style="margin-bottom: 10px;" />
                    <button type="button" class="button" id="ffldb_test_check"><?php esc_html_e('Check FFL', 'ffl-db-connector'); ?></button>
                    <div id="ffldb_test_result"></div>
                </td>
            </tr>
        </table>
    </div>
    <?php endif; ?>
</div>

<style>
.card {
    background: #fff;
    border: 1px solid #ccd0d4;
    padding: 20px;
    margin-top: 20px;
    box-shadow: 0 1px 1px rgba(0,0,0,0.04);
}

#ffldb_test_result {
    margin-top: 10px;
    padding: 10px;
    background: #f0f0f0;
    border-radius: 3px;
    display: none;
}

#ffldb_test_result.success {
    background: #d4edda;
    color: #155724;
    border: 1px solid #c3e6cb;
}

#ffldb_test_result.error {
    background: #f8d7da;
    color: #721c24;
    border: 1px solid #f5c6cb;
}

#ffldb_verify_status {
    margin-top: 10px;
}
</style>

<script type="text/javascript">
jQuery(document).ready(function($) {
    // Toggle API key visibility
    $('#ffldb_toggle_key').on('click', function() {
        var input = $('#ffldb_api_key');
        var button = $(this);
        
        if (input.attr('type') === 'password') {
            input.attr('type', 'text');
            button.text('<?php echo esc_js(__('Hide', 'ffl-db-connector')); ?>');
        } else {
            input.attr('type', 'password');
            button.text('<?php echo esc_js(__('Show', 'ffl-db-connector')); ?>');
        }
    });
    
    // Verify API key
    $('#ffldb_verify_key').on('click', function() {
        var button = $(this);
        var apiKey = $('#ffldb_api_key').val();
        var status = $('#ffldb_verify_status');
        
        if (!apiKey) {
            status.html('<div class="notice notice-error inline"><p><?php echo esc_js(__('Please enter an API key', 'ffl-db-connector')); ?></p></div>');
            return;
        }
        
        button.prop('disabled', true).text('<?php echo esc_js(__('Verifying...', 'ffl-db-connector')); ?>');
        status.html('');
        
        $.post(ajaxurl, {
            action: 'ffldb_verify_api',
            nonce: '<?php echo wp_create_nonce('ffldb_verify'); ?>',
            api_key: apiKey
        }, function(response) {
            button.prop('disabled', false).text('<?php echo esc_js(__('Verify API Key', 'ffl-db-connector')); ?>');
            
            if (response.success) {
                var msg = '<div class="notice notice-success inline"><p>✓ ' + response.data.message + '</p>';
                if (response.data.account_info) {
                    msg += '<p><strong><?php echo esc_js(__('Requests Today:', 'ffl-db-connector')); ?></strong> ' + response.data.account_info.requests_today;
                    msg += ' | <strong><?php echo esc_js(__('Remaining:', 'ffl-db-connector')); ?></strong> ' + response.data.account_info.requests_remaining + '</p>';
                }
                msg += '</div>';
                status.html(msg);
                
                // Reload page to show settings
                setTimeout(function() {
                    location.reload();
                }, 1500);
            } else {
                status.html('<div class="notice notice-error inline"><p>✗ ' + response.data + '</p></div>');
            }
        }).fail(function() {
            button.prop('disabled', false).text('<?php echo esc_js(__('Verify API Key', 'ffl-db-connector')); ?>');
            status.html('<div class="notice notice-error inline"><p><?php echo esc_js(__('Connection error', 'ffl-db-connector')); ?></p></div>');
        });
    });
    
    // Clear cache
    $('#ffldb_clear_cache').on('click', function() {
        var button = $(this);
        
        if (!confirm('<?php echo esc_js(__('Are you sure you want to clear all cached FFL data?', 'ffl-db-connector')); ?>')) {
            return;
        }
        
        button.prop('disabled', true).text('<?php echo esc_js(__('Clearing...', 'ffl-db-connector')); ?>');
        
        $.post(ajaxurl, {
            action: 'ffldb_clear_cache',
            nonce: '<?php echo wp_create_nonce('ffldb_cache'); ?>'
        }, function(response) {
            button.prop('disabled', false).text('<?php echo esc_js(__('Force Refresh All Cached Data', 'ffl-db-connector')); ?>');
            
            if (response.success) {
                alert(response.data);
                location.reload();
            } else {
                alert('<?php echo esc_js(__('Error:', 'ffl-db-connector')); ?> ' + response.data);
            }
        });
    });
    
    // Test FFL lookup
    $('#ffldb_test_check').on('click', function() {
        var button = $(this);
        var fflNumber = $('#ffldb_test_number').val();
        var result = $('#ffldb_test_result');
        
        if (!fflNumber) {
            result.removeClass('success').addClass('error').html('<?php echo esc_js(__('Please enter an FFL number', 'ffl-db-connector')); ?>').show();
            return;
        }
        
        button.prop('disabled', true).text('<?php echo esc_js(__('Checking...', 'ffl-db-connector')); ?>');
        result.removeClass('success error').html('<?php echo esc_js(__('Checking FFL status...', 'ffl-db-connector')); ?>').show();
        
        $.post(ajaxurl, {
            action: 'ffldb_check_ffl',
            nonce: '<?php echo wp_create_nonce('ffldb_check'); ?>',
            ffl_number: fflNumber
        }, function(response) {
            button.prop('disabled', false).text('<?php echo esc_js(__('Check FFL', 'ffl-db-connector')); ?>');
            
            if (response.success) {
                var data = response.data;
                var html = '';
                
                if (data.on_file) {
                    result.removeClass('error').addClass('success');
                    html = '<strong>✓ <?php echo esc_js(__('ON FILE', 'ffl-db-connector')); ?></strong><br>';
                    html += '<?php echo esc_js(__('URL:', 'ffl-db-connector')); ?> <a href="' + data.url + '" target="_blank">' + data.url + '</a>';
                    if (data.expiration) {
                        html += '<br><strong><?php echo esc_js(__('Expires:', 'ffl-db-connector')); ?></strong> ' + data.expiration;
                        if (data.expiration_timestamp) {
                            var daysUntil = Math.floor((data.expiration_timestamp - (Date.now() / 1000)) / 86400);
                            if (daysUntil <= 60) {
                                html += ' <span style="color: #f59e0b; font-weight: bold;">(' + daysUntil + ' <?php echo esc_js(__('days remaining', 'ffl-db-connector')); ?>)</span>';
                            }
                        }
                    }
                    if (data.business_name) {
                        html += '<br><?php echo esc_js(__('Business:', 'ffl-db-connector')); ?> ' + data.business_name;
                    }
                } else {
                    result.removeClass('success').addClass('error');
                    html = '<strong>✗ <?php echo esc_js(__('NOT ON FILE', 'ffl-db-connector')); ?></strong>';
                    if (data.expired) {
                        html += '<br><span style="color: #dc2626; font-weight: bold;"><?php echo esc_js(__('License has expired', 'ffl-db-connector')); ?></span>';
                        if (data.expiration) {
                            html += ' (<?php echo esc_js(__('Expired:', 'ffl-db-connector')); ?> ' + data.expiration + ')';
                        }
                    }
                    if (data.error) {
                        html += '<br><?php echo esc_js(__('Error:', 'ffl-db-connector')); ?> ' + data.error;
                    }
                }
                
                result.html(html);
            } else {
                result.removeClass('success').addClass('error').html('<?php echo esc_js(__('Error:', 'ffl-db-connector')); ?> ' + response.data);
            }
        }).fail(function() {
            button.prop('disabled', false).text('<?php echo esc_js(__('Check FFL', 'ffl-db-connector')); ?>');
            result.removeClass('success').addClass('error').html('<?php echo esc_js(__('Connection error', 'ffl-db-connector')); ?>');
        });
    });
});
</script>
