<?php
/**
 * Plugin Name: FFL-DB Connector for WooCommerce
 * Plugin URI: https://ffl-db.com
 * Description: Integrates with FFL-DB.com to share FFLs between dealers
 * Version: 1.0.0
 * Author: Etowah Firearms
 * Author URI: https://ffl-db.com
 * Text Domain: ffl-db-connector
 * Domain Path: /languages
 * Requires at least: 5.8
 * Requires PHP: 7.4
 * WC requires at least: 6.0
 * WC tested up to: 8.0
 * License: GPL-2.0+
 * License URI: http://www.gnu.org/licenses/gpl-2.0.txt
 */

if (!defined('ABSPATH')) {
    exit;
}

// Define plugin constants
define('FFLDB_VERSION', '1.0.0');
define('FFLDB_PLUGIN_DIR', plugin_dir_path(__FILE__));
define('FFLDB_PLUGIN_URL', plugin_dir_url(__FILE__));
define('FFLDB_PLUGIN_BASENAME', plugin_basename(__FILE__));

/**
 * Main FFL-DB Connector Class
 * 
 * This is the primary class for the plugin. It handles:
 * - API connection to FFL-DB.com
 * - Settings management
 * - Caching of FFL lookups
 * - Admin interface
 * 
 * @since 1.0.0
 */
class FFLDB_Connector {
    
    /**
     * API endpoint for FFL-DB.com
     * Uses WordPress REST API structure
     */
    const API_BASE_URL = 'https://ffl-db.com/wp-json/ffl/v1/';
    
    /**
     * Option keys for settings
     */
    const OPTION_API_KEY = 'ffldb_api_key';
    const OPTION_API_VERIFIED = 'ffldb_api_verified';
    const OPTION_SETTINGS = 'ffldb_settings';
    const OPTION_CACHE_PREFIX = 'ffldb_cache_';
    
    /**
     * Singleton instance
     * @var FFLDB_Connector
     */
    private static $instance = null;
    
    /**
     * Get singleton instance
     * 
     * @return FFLDB_Connector
     */
    public static function get_instance() {
        if (null === self::$instance) {
            self::$instance = new self();
        }
        return self::$instance;
    }
    
    /**
     * Constructor
     */
    private function __construct() {
        $this->init_hooks();
    }
    
    /**
     * Initialize WordPress hooks
     */
    private function init_hooks() {
        // Admin hooks
        add_action('admin_menu', array($this, 'add_admin_menu'));
        add_action('admin_enqueue_scripts', array($this, 'enqueue_admin_scripts'));
        
        // AJAX handlers
        add_action('wp_ajax_ffldb_verify_api', array($this, 'ajax_verify_api_key'));
        add_action('wp_ajax_ffldb_check_ffl', array($this, 'ajax_check_ffl'));
        add_action('wp_ajax_ffldb_clear_cache', array($this, 'ajax_clear_cache'));
        
        // Public AJAX (for checkout connector)
        add_action('wp_ajax_nopriv_ffldb_check_ffl_public', array($this, 'ajax_check_ffl_public'));
        add_action('wp_ajax_ffldb_check_ffl_public', array($this, 'ajax_check_ffl_public'));
        
        // Add settings link to plugins page
        add_filter('plugin_action_links_' . FFLDB_PLUGIN_BASENAME, array($this, 'add_settings_link'));
        
        // Activation/Deactivation hooks
        register_activation_hook(__FILE__, array($this, 'activate'));
        register_deactivation_hook(__FILE__, array($this, 'deactivate'));
    }
    
    /**
     * Plugin activation
     */
    public function activate() {
        $this->log('FFL-DB Connector plugin activated');
    }
    
    /**
     * Plugin deactivation
     */
    public function deactivate() {
        // Optionally clear cache on deactivation
        $this->clear_all_cache();
        $this->log('FFL-DB Connector plugin deactivated');
    }
    
    /**
     * Add settings link to plugins page
     */
    public function add_settings_link($links) {
        $settings_link = '<a href="' . admin_url('admin.php?page=ffldb-connector') . '">' . __('Settings', 'ffl-db-connector') . '</a>';
        array_unshift($links, $settings_link);
        return $links;
    }
    
    /**
     * Check if API key is verified
     * 
     * @return bool
     */
    public function is_api_verified() {
        return (bool) get_option(self::OPTION_API_VERIFIED, false);
    }
    
    /**
     * Get API key
     * 
     * @return string
     */
    public function get_api_key() {
        return get_option(self::OPTION_API_KEY, '');
    }
    
    /**
     * Get plugin settings (internal)
     * 
     * @return array
     */
    private function get_settings() {
        return array(
            'cache_not_found_hours' => 4, // Hardcoded to 4 hours
        );
    }
    
    /**
     * ================================================================
     * PUBLIC API METHODS
     * These methods can be called from your theme or other plugins
     * ================================================================
     */
    
    /**
     * Check if an FFL license is on file with FFL-DB.com
     * 
     * This is the main public API method. Use this to check if an FFL
     * dealer has their license on file with FFL-DB.com.
     * 
     * @param string $ffl_number The FFL license number (format: X-XX-XXX-XX-XX-XXXXX)
     * @param bool   $bypass_cache Whether to bypass the cache and make a fresh API call
     * 
     * @return array {
     *     @type bool        $on_file              Whether the FFL is on file
     *     @type string|null $url                  URL to the FFL on FFL-DB.com
     *     @type string|null $expiration           Expiration date formatted (e.g., "January 1, 2026")
     *     @type int|null    $expiration_timestamp Expiration date as Unix timestamp
     *     @type string|null $last_updated         When the record was last updated
     *     @type string|null $business_name        Business name (if available)
     *     @type string|null $error                Error message if the check failed
     *     @type bool        $expired              True if the license has expired
     *     @type string      $cached_at            When this result was cached
     * }
     * 
     * @example
     * // Basic usage
     * $ffldb = FFLDB_Connector::get_instance();
     * $result = $ffldb->check_ffl_on_file('1-54-001-01-5K-04379');
     * 
     * if ($result['on_file']) {
     *     echo 'FFL is on file!';
     *     echo 'View: ' . $result['url'];
     *     echo 'Expires: ' . $result['expiration'];
     * } else {
     *     if ($result['expired']) {
     *         echo 'FFL license has expired.';
     *     } else {
     *         echo 'FFL is NOT on file.';
     *     }
     * }
     */
    public function check_ffl_on_file($ffl_number, $bypass_cache = false) {
        if (!$this->is_api_verified()) {
            return array(
                'on_file' => false,
                'error' => 'API not configured'
            );
        }
        
        // Normalize FFL number
        $ffl_number = $this->normalize_ffl_number($ffl_number);
        
        if (empty($ffl_number)) {
            return array(
                'on_file' => false,
                'error' => 'Invalid FFL number format'
            );
        }
        
        // Check cache first (unless bypassing)
        if (!$bypass_cache) {
            $cached = $this->get_cached_result($ffl_number);
            if ($cached !== false) {
                return $cached;
            }
        }
        
        // Make API request
        return $this->perform_ffl_lookup($ffl_number);
    }
    
    /**
     * Check multiple FFL licenses in batch
     * 
     * More efficient than calling check_ffl_on_file() multiple times
     * as it uses cached results where available.
     * 
     * @param array $ffl_numbers Array of FFL license numbers
     * 
     * @return array Associative array of FFL number => result
     * 
     * @example
     * $ffldb = FFLDB_Connector::get_instance();
     * $results = $ffldb->check_ffls_batch(array(
     *     '1-54-001-01-5K-04379',
     *     '1-54-002-02-6K-12345'
     * ));
     * 
     * foreach ($results as $ffl_number => $status) {
     *     echo $ffl_number . ': ' . ($status['on_file'] ? 'ON FILE' : 'NOT ON FILE');
     * }
     */
    public function check_ffls_batch($ffl_numbers) {
        $results = array();
        
        foreach ($ffl_numbers as $ffl_number) {
            $ffl_number = $this->normalize_ffl_number($ffl_number);
            if (!empty($ffl_number)) {
                $results[$ffl_number] = $this->check_ffl_on_file($ffl_number);
            }
        }
        
        return $results;
    }
    
    /**
     * Get the FFL-DB.com URL for an FFL license
     * 
     * @param string $ffl_number The FFL license number
     * 
     * @return string URL to the FFL on FFL-DB.com
     * 
     * @example
     * $ffldb = FFLDB_Connector::get_instance();
     * $url = $ffldb->get_ffl_url('1-54-001-01-5K-04379');
     * echo '<a href="' . $url . '" target="_blank">View on FFL-DB</a>';
     */
    public function get_ffl_url($ffl_number) {
        $ffl_number = $this->normalize_ffl_number($ffl_number);
        return 'https://ffl-db.com/ffl/' . $ffl_number;
    }
    
    /**
     * Test the API connection with the current API key
     * 
     * @param string|null $api_key Optional API key to test (uses saved key if null)
     * 
     * @return array {
     *     @type bool   $success      Whether the connection succeeded
     *     @type string $message      Status message
     *     @type array  $account_info Account information (requests_today, requests_remaining, etc.)
     * }
     * 
     * @example
     * $ffldb = FFLDB_Connector::get_instance();
     * $test = $ffldb->test_api_connection();
     * 
     * if ($test['success']) {
     *     echo 'Connected! Requests remaining: ' . $test['account_info']['requests_remaining'];
     * }
     */
    public function test_api_connection($api_key = null) {
        if ($api_key === null) {
            $api_key = $this->get_api_key();
        }
        
        if (empty($api_key)) {
            return array(
                'success' => false,
                'message' => 'API key is required'
            );
        }
        
        return $this->perform_api_test($api_key);
    }
    
    /**
     * Clear the cache for a specific FFL number
     * 
     * @param string $ffl_number The FFL license number
     * 
     * @return bool True if cache was cleared
     */
    public function clear_ffl_cache($ffl_number) {
        $ffl_number = $this->normalize_ffl_number($ffl_number);
        $cache_key = self::OPTION_CACHE_PREFIX . md5($ffl_number);
        return delete_transient($cache_key);
    }
    
    /**
     * Clear all cached FFL results
     * 
     * @return int Number of cache entries cleared
     */
    public function clear_all_cache() {
        global $wpdb;
        
        $count = $wpdb->query(
            "DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_" . self::OPTION_CACHE_PREFIX . "%'"
        );
        $wpdb->query(
            "DELETE FROM {$wpdb->options} WHERE option_name LIKE '_transient_timeout_" . self::OPTION_CACHE_PREFIX . "%'"
        );
        
        return $count;
    }
    
    /**
     * ================================================================
     * PRIVATE/INTERNAL METHODS
     * ================================================================
     */
    
    /**
     * Normalize FFL number format
     */
    private function normalize_ffl_number($ffl_number) {
        // Remove any whitespace
        $ffl_number = trim($ffl_number);
        
        // Basic validation - should contain hyphens and alphanumeric
        if (empty($ffl_number) || !preg_match('/^[\d\-A-Za-z]+$/', $ffl_number)) {
            return '';
        }
        
        return strtoupper($ffl_number);
    }
    
    /**
     * Get cached result for FFL number
     */
    private function get_cached_result($ffl_number) {
        $cache_key = self::OPTION_CACHE_PREFIX . md5($ffl_number);
        $cached = get_transient($cache_key);
        
        if ($cached === false) {
            return false;
        }
        
        // Check if we have an expiration timestamp and if it's passed
        $expiration_time = null;
        if (!empty($cached['expiration_timestamp'])) {
            $expiration_time = (int) $cached['expiration_timestamp'];
        } elseif (!empty($cached['expiration'])) {
            $expiration_time = strtotime($cached['expiration']);
        }
        
        if ($expiration_time && $expiration_time < time()) {
            // FFL has expired, update cache to reflect this
            $this->log('Cached FFL ' . $ffl_number . ' has expired');
            $cached['on_file'] = false;
            $cached['expired'] = true;
            set_transient($cache_key, $cached, 0);
        }
        
        $this->log('Using cached result for FFL: ' . $ffl_number);
        return $cached;
    }
    
    /**
     * Perform the actual FFL lookup via API
     */
    private function perform_ffl_lookup($ffl_number) {
        $api_key = $this->get_api_key();
        $endpoint = self::API_BASE_URL . 'lookup/' . urlencode($ffl_number);
        
        $this->log('Checking FFL: ' . $ffl_number . ' at endpoint: ' . $endpoint);
        
        // Try with X-API-Key header (preferred method)
        $response = wp_remote_get(
            $endpoint,
            array(
                'headers' => array(
                    'X-API-Key' => $api_key,
                    'Accept' => 'application/json',
                ),
                'timeout' => 10,
            )
        );
        
        // If that fails, try with query parameter
        if (is_wp_error($response) || wp_remote_retrieve_response_code($response) === 401) {
            $this->log('Trying with query parameter authentication');
            $response = wp_remote_get(
                $endpoint . '?api_key=' . urlencode($api_key),
                array(
                    'headers' => array(
                        'Accept' => 'application/json',
                    ),
                    'timeout' => 10,
                )
            );
        }
        
        if (is_wp_error($response)) {
            $this->log('API error for FFL ' . $ffl_number . ': ' . $response->get_error_message());
            return array(
                'on_file' => false,
                'error' => 'Connection error: ' . $response->get_error_message()
            );
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        
        $this->log('API Response Code for FFL lookup: ' . $status_code);
        
        // Parse the response
        return $this->parse_ffl_response($ffl_number, $status_code, $body);
    }
    
    /**
     * Parse the API response for FFL lookup
     */
    private function parse_ffl_response($ffl_number, $status_code, $body) {
        $cache_key = self::OPTION_CACHE_PREFIX . md5($ffl_number);
        $settings = $this->get_settings();
        $cache_hours = isset($settings['cache_not_found_hours']) ? intval($settings['cache_not_found_hours']) : 4;
        
        $result = array(
            'on_file' => false,
            'url' => null,
            'last_updated' => null,
            'expiration' => null,
            'expiration_timestamp' => null,
            'expired' => false,
            'cached_at' => current_time('mysql'),
        );
        
        // Try to decode JSON response
        $data = json_decode($body, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            $this->log('JSON decode error for FFL ' . $ffl_number . ': ' . json_last_error_msg());
            
            // If it's a 404, treat as not on file and cache
            if ($status_code === 404) {
                $result['not_on_file_cached_at'] = current_time('mysql');
                set_transient($cache_key, $result, $cache_hours * HOUR_IN_SECONDS);
                return $result;
            }
            
            return array(
                'on_file' => false,
                'error' => 'Invalid response format'
            );
        }
        
        // Handle the response based on the API structure
        if ($status_code === 200) {
            // Check if FFL is expired from API response
            $is_expired = isset($data['is_expired']) ? (bool) $data['is_expired'] : false;
            
            // Get expiration timestamp from API
            $expiration_timestamp = null;
            if (isset($data['expiration_date']) && is_numeric($data['expiration_date'])) {
                $expiration_timestamp = (int) $data['expiration_date'];
                $result['expiration_timestamp'] = $expiration_timestamp;
                $result['expiration'] = date('F j, Y', $expiration_timestamp);
                
                // Double-check expiration locally
                if ($expiration_timestamp < time()) {
                    $is_expired = true;
                }
            }
            
            $result['expired'] = $is_expired;
            
            // Determine on_file status (API already excludes expired FFLs)
            $ffl_on_file = $this->determine_on_file_status($data);
            
            if ($ffl_on_file && !$is_expired) {
                $result['on_file'] = true;
                $result['url'] = $this->get_ffl_url($ffl_number);
                
                // Extract FFL details if available
                if (!empty($data['ffl'])) {
                    $ffl_data = $data['ffl'];
                    $result['last_updated'] = $ffl_data['updated_at'] ?? null;
                    $result['business_name'] = $ffl_data['business_name'] ?? null;
                }
            }
        } elseif ($status_code === 404) {
            $result['not_on_file_cached_at'] = current_time('mysql');
            set_transient($cache_key, $result, $cache_hours * HOUR_IN_SECONDS);
            return $result;
        } elseif ($status_code === 401) {
            return array(
                'on_file' => false,
                'error' => 'Authentication failed - please check API key'
            );
        } else {
            return array(
                'on_file' => false,
                'error' => 'API returned status ' . $status_code
            );
        }
        
        // Cache the result
        $this->cache_result($ffl_number, $result);
        
        return $result;
    }
    
    /**
     * Determine if FFL is on file from API response data
     */
    private function determine_on_file_status($data) {
        // Check various indicators that the FFL has a document on file
        
        if (!empty($data['ffl']) && is_array($data['ffl'])) {
            $ffl = $data['ffl'];
            
            // Explicit 'on_file' or 'has_document' field
            if (isset($ffl['on_file']) && $ffl['on_file'] === true) {
                return true;
            }
            
            // 'status' field indicates on file
            if (isset($ffl['status']) && $ffl['status'] === 'on_file') {
                return true;
            }
            
            // Has document_url or file_url
            if (!empty($ffl['document_url']) || !empty($ffl['file_url'])) {
                return true;
            }
            
            // Has 'verified' or 'has_copy' flag
            if ((isset($ffl['verified']) && $ffl['verified'] === true) ||
                (isset($ffl['has_copy']) && $ffl['has_copy'] === true)) {
                return true;
            }
        }
        
        // Top-level 'on_file' field
        if (isset($data['on_file']) && $data['on_file'] === true) {
            return true;
        }
        
        // Alternative response format
        if (isset($data['found']) && $data['found'] === true && 
            isset($data['on_file']) && $data['on_file'] === true) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Cache the FFL lookup result
     */
    private function cache_result($ffl_number, $result) {
        $cache_key = self::OPTION_CACHE_PREFIX . md5($ffl_number);
        $settings = $this->get_settings();
        $cache_hours = isset($settings['cache_not_found_hours']) ? intval($settings['cache_not_found_hours']) : 4;
        
        if ($result['on_file']) {
            $cache_expiration = 0; // Default to no expiration (permanent cache)
            
            // Use Unix timestamp if available, otherwise try to parse string date
            $expiration_time = null;
            if (!empty($result['expiration_timestamp'])) {
                $expiration_time = (int) $result['expiration_timestamp'];
            } elseif (!empty($result['expiration'])) {
                $expiration_time = strtotime($result['expiration']);
            }
            
            if ($expiration_time) {
                $days_until_expiration = ($expiration_time - time()) / 86400;
                
                if ($days_until_expiration > 0) {
                    // Cache until 1 day after expiration
                    $cache_expiration = $expiration_time + 86400;
                } else {
                    // FFL is already expired
                    $result['on_file'] = false;
                    $result['expired'] = true;
                }
            }
            
            if ($cache_expiration > 0) {
                set_transient($cache_key, $result, $cache_expiration - time());
            } else {
                set_transient($cache_key, $result, 0);
            }
        } else {
            // Not on file - cache for configured hours
            $result['not_on_file_cached_at'] = current_time('mysql');
            set_transient($cache_key, $result, $cache_hours * HOUR_IN_SECONDS);
        }
    }
    
    /**
     * Perform API connection test
     */
    private function perform_api_test($api_key) {
        $this->log('Testing API connection with key: ' . substr($api_key, 0, 10) . '...');
        
        // Use the status endpoint to verify API key
        $response = wp_remote_get(
            self::API_BASE_URL . 'status',
            array(
                'headers' => array(
                    'X-API-Key' => $api_key,
                    'Accept' => 'application/json',
                ),
                'timeout' => 15,
                'sslverify' => true,
            )
        );
        
        if (is_wp_error($response)) {
            // Try with query parameter as fallback
            $response = wp_remote_get(
                self::API_BASE_URL . 'status?api_key=' . urlencode($api_key),
                array(
                    'headers' => array(
                        'Accept' => 'application/json',
                    ),
                    'timeout' => 15,
                    'sslverify' => true,
                )
            );
            
            if (is_wp_error($response)) {
                return array(
                    'success' => false,
                    'message' => 'Connection error: ' . $response->get_error_message()
                );
            }
        }
        
        $status_code = wp_remote_retrieve_response_code($response);
        $body = wp_remote_retrieve_body($response);
        $data = json_decode($body, true);
        
        if (json_last_error() !== JSON_ERROR_NONE) {
            if (strpos($body, '<html') !== false || strpos($body, '<!DOCTYPE') !== false) {
                return array(
                    'success' => false,
                    'message' => 'API returned HTML instead of JSON. Status: ' . $status_code
                );
            }
            
            return array(
                'success' => false,
                'message' => 'Invalid API response format. Status code: ' . $status_code
            );
        }
        
        if ($status_code === 200 || $status_code === 201) {
            return array(
                'success' => true,
                'message' => 'API key is valid',
                'account_info' => array(
                    'requests_today' => $data['requests_today'] ?? 0,
                    'requests_remaining' => $data['requests_remaining'] ?? 0,
                    'daily_limit' => $data['daily_limit'] ?? 0,
                    'status' => $data['status'] ?? 'active'
                )
            );
        } elseif ($status_code === 401 || $status_code === 403) {
            $error_message = $data['message'] ?? 'Authentication failed';
            return array(
                'success' => false,
                'message' => $error_message
            );
        } else {
            return array(
                'success' => false,
                'message' => 'API error: HTTP ' . $status_code
            );
        }
    }
    
    /**
     * Log messages when WP_DEBUG is enabled
     */
    private function log($message) {
        if (defined('WP_DEBUG') && WP_DEBUG) {
            error_log('FFL-DB Connector: ' . $message);
        }
    }
    
    /**
     * ================================================================
     * AJAX HANDLERS
     * ================================================================
     */
    
    /**
     * AJAX: Verify API key
     */
    public function ajax_verify_api_key() {
        check_ajax_referer('ffldb_verify', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
            return;
        }
        
        $api_key = sanitize_text_field($_POST['api_key'] ?? '');
        
        if (empty($api_key)) {
            wp_send_json_error('API key is required');
            return;
        }
        
        $test_result = $this->test_api_connection($api_key);
        
        if ($test_result['success']) {
            update_option(self::OPTION_API_KEY, $api_key);
            update_option(self::OPTION_API_VERIFIED, true);
            
            wp_send_json_success(array(
                'message' => 'API key verified successfully',
                'account_info' => $test_result['account_info'] ?? null
            ));
        } else {
            update_option(self::OPTION_API_VERIFIED, false);
            wp_send_json_error($test_result['message']);
        }
    }
    
    /**
     * AJAX: Check single FFL (admin)
     */
    public function ajax_check_ffl() {
        check_ajax_referer('ffldb_check', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
            return;
        }
        
        $ffl_number = sanitize_text_field($_POST['ffl_number'] ?? '');
        
        if (empty($ffl_number)) {
            wp_send_json_error('FFL number is required');
            return;
        }
        
        $result = $this->check_ffl_on_file($ffl_number);
        
        wp_send_json_success($result);
    }
    
    /**
     * AJAX: Check FFL (public/frontend)
     */
    public function ajax_check_ffl_public() {
        // Verify nonce if provided
        if (!empty($_POST['nonce'])) {
            if (!wp_verify_nonce($_POST['nonce'], 'ffldb_public_check')) {
                wp_send_json_error('Security check failed');
                return;
            }
        }
        
        $ffl_number = sanitize_text_field($_POST['ffl_number'] ?? '');
        
        if (empty($ffl_number)) {
            wp_send_json_error('FFL number is required');
            return;
        }
        
        $result = $this->check_ffl_on_file($ffl_number);
        
        // Only return minimal info for public requests
        wp_send_json_success(array(
            'on_file' => $result['on_file'],
            'expired' => $result['expired'] ?? false
        ));
    }
    
    /**
     * AJAX: Clear cache
     */
    public function ajax_clear_cache() {
        check_ajax_referer('ffldb_cache', 'nonce');
        
        if (!current_user_can('manage_woocommerce')) {
            wp_send_json_error('Insufficient permissions');
            return;
        }
        
        $cleared = $this->clear_all_cache();
        
        wp_send_json_success('Cache cleared successfully (' . $cleared . ' entries)');
    }
    
    /**
     * ================================================================
     * ADMIN UI
     * ================================================================
     */
    
    /**
     * Add admin menu
     */
    public function add_admin_menu() {
        add_submenu_page(
            'woocommerce',
            __('FFL-DB Connector', 'ffl-db-connector'),
            __('FFL-DB Connector', 'ffl-db-connector'),
            'manage_woocommerce',
            'ffldb-connector',
            array($this, 'render_admin_page')
        );
    }
    
    /**
     * Enqueue admin scripts
     */
    public function enqueue_admin_scripts($hook) {
        if ($hook !== 'woocommerce_page_ffldb-connector') {
            return;
        }
        
        wp_enqueue_script('jquery');
    }
    
    /**
     * Render admin page
     */
    public function render_admin_page() {
        $api_key = $this->get_api_key();
        $is_verified = $this->is_api_verified();
        $settings = $this->get_settings();
        
        // Include the admin page template
        include FFLDB_PLUGIN_DIR . 'templates/admin-page.php';
    }
}

/**
 * ================================================================
 * GLOBAL HELPER FUNCTIONS
 * These make it easy to use the API from anywhere
 * ================================================================
 */

/**
 * Get the FFL-DB Connector instance
 * 
 * @return FFLDB_Connector
 * 
 * @example
 * $ffldb = ffldb_get_instance();
 * $result = $ffldb->check_ffl_on_file('1-54-001-01-5K-04379');
 */
function ffldb_get_instance() {
    return FFLDB_Connector::get_instance();
}

/**
 * Check if an FFL is on file with FFL-DB.com
 * 
 * Shortcut function for quick checks.
 * 
 * @param string $ffl_number The FFL license number
 * @param bool   $bypass_cache Whether to bypass the cache
 * 
 * @return array Result array with on_file status
 * 
 * @example
 * $result = ffldb_check_ffl('1-54-001-01-5K-04379');
 * if ($result['on_file']) {
 *     echo 'This FFL is on file!';
 * }
 */
function ffldb_check_ffl($ffl_number, $bypass_cache = false) {
    return ffldb_get_instance()->check_ffl_on_file($ffl_number, $bypass_cache);
}

/**
 * Check if an FFL is on file (returns boolean only)
 * 
 * Simple function that returns just true/false.
 * 
 * @param string $ffl_number The FFL license number
 * 
 * @return bool True if on file, false otherwise
 * 
 * @example
 * if (ffldb_is_on_file('1-54-001-01-5K-04379')) {
 *     echo 'On file!';
 * }
 */
function ffldb_is_on_file($ffl_number) {
    $result = ffldb_check_ffl($ffl_number);
    return !empty($result['on_file']);
}

/**
 * Get the FFL-DB.com URL for an FFL
 * 
 * @param string $ffl_number The FFL license number
 * 
 * @return string URL to view the FFL on FFL-DB.com
 * 
 * @example
 * echo '<a href="' . ffldb_get_url('1-54-001-01-5K-04379') . '">View on FFL-DB</a>';
 */
function ffldb_get_url($ffl_number) {
    return ffldb_get_instance()->get_ffl_url($ffl_number);
}

/**
 * Check if the FFL-DB API is connected and verified
 * 
 * @return bool True if API is verified
 * 
 * @example
 * if (ffldb_is_connected()) {
 *     echo 'FFL-DB is connected!';
 * }
 */
function ffldb_is_connected() {
    return ffldb_get_instance()->is_api_verified();
}

// Declare HPOS compatibility
add_action('before_woocommerce_init', function() {
    if (class_exists('\Automattic\WooCommerce\Utilities\FeaturesUtil')) {
        \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility('custom_order_tables', __FILE__, true);
    }
});

// Initialize the plugin
add_action('plugins_loaded', function() {
    // Make sure WooCommerce is active
    if (!class_exists('WooCommerce')) {
        add_action('admin_notices', function() {
            echo '<div class="error"><p><strong>FFL-DB Connector</strong> requires WooCommerce to be installed and active.</p></div>';
        });
        return;
    }
    
    FFLDB_Connector::get_instance();
});
